
//
//    Copyright  2010, 2011 Thomas C. McDermott, N5EG
//    This file is part of ABCDmatrix - the 2-Port Network Calculator program.
//
//    ABCDmatrix is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    ABCDmatrix is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with ABCDmatrix, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


using System;

namespace ABCDmatrix
{
    /// <summary>
    /// Complex double type for 2-port analysis.
    /// </summary>
    [Serializable]
    public class Complex
    {
        /// <summary>
        /// Real part of the complex number
        /// </summary>
        public Double real;
        /// <summary>
        /// Imaginary part of the complex number
        /// </summary>
        public Double imag;
        
        // useful constants for 2-port analysis
        /// <summary>
        /// Zero is the constant Complex(0,0)
        /// </summary>
        public static readonly Complex Zero = new Complex();
        /// <summary>
        /// One is the constant Complex(1,0)
        /// </summary>
        public static readonly Complex One = new Complex(1);
        /// <summary>
        /// Two is the constant Complex(2,0)
        /// </summary>
        public static readonly Complex Two = new Complex(2);
        /// <summary>
        /// J is the constant Complex(0,1)
        /// </summary>
        public static readonly Complex J = new Complex(0, 1);

        public Complex(Double real, Double imag)
        {
            this.real = real;
            this.imag = imag;
        }
        public Complex(Single real, Single imag)
        {
            this.real = Convert.ToDouble(real);
            this.imag = Convert.ToDouble(imag);
        }
        public Complex(Int32 real, Int32 imag)
        {
            this.real = Convert.ToDouble(real);
            this.imag = Convert.ToDouble(imag);
        }
        public Complex()        // default constructor
        {
            //this.real = 0;    // already initialized by compiler
            //this.imag = 0;
        }
        public Complex(Double real) //unary constructor
        {
            this.real = real;
            //this.imag = 0;
        }
        public Complex(Single real) //unary constructor
        {
            this.real = Convert.ToDouble(real);
            //this.imag = 0;
        }
        public Complex(Int32 real) //unary constructor
        {
            this.real = Convert.ToDouble(real);
            //this.imag = 0;
        }
        public Complex(Complex c1)  // assignment constructor (deep copy)
        {
            this.real = c1.real;
            this.imag = c1.imag;
        }

        /// <summary>
        /// Complex addition
        /// </summary>
        /// <param name="c1"></param>
        /// <param name="c2"></param>
        /// <returns></returns>
        public static Complex operator +(Complex c1, Complex c2)
        {
            return new Complex(c1.real + c2.real, c1.imag + c2.imag);
        }

        /// <summary>
        /// Complex subtraction
        /// </summary>
        /// <param name="c1"></param>
        /// <param name="c2"></param>
        /// <returns></returns>
        public static Complex operator -(Complex c1, Complex c2)
        {
            return new Complex(c1.real - c2.real, c1.imag - c2.imag);
        }

        /// <summary>
        /// Unary negation
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex operator -(Complex c1)
        {
            return new Complex(-c1.real, -c1.imag);
        }

        /// <summary>
        /// Complex multiplication
        /// </summary>
        /// <param name="c1"></param>
        /// <param name="c2"></param>
        /// <returns></returns>
        public static Complex operator *(Complex c1, Complex c2)
        {
            return new Complex(c1.real * c2.real - c1.imag * c2.imag,
                c1.real * c2.imag + c1.imag * c2.real);
        }

        /// <summary>
        /// Complex conjugate
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex Conjugate(Complex c1)
        {
            return new Complex(c1.real, -c1.imag);
        }

        /// <summary>
        /// Length of the complex vector
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Double Magnitude(Complex c1)
        {
            return Math.Sqrt(c1.real * c1.real + c1.imag * c1.imag); 
        }

        /// <summary>
        /// Magnitude (length) of the complex vector expressed as voltage in dB.
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Double MagnitudeDB(Complex c1)
        {
            return 20 * Math.Log10(Magnitude(c1));
        }

        /// <summary>
        /// Angle in radians
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Double ArgRad(Complex c1)
        {
            return Math.Atan2(c1.imag, c1.real);
        }

        /// <summary>
        /// Angle in degrees
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Double ArgDegr(Complex c1)
        {
            return (180 / Math.PI) * ArgRad(c1);
        }

        /// <summary>
        /// Complex division
        /// </summary>
        /// <param name="c1"></param>
        /// <param name="c2"></param>
        /// <returns></returns>
        public static Complex operator /(Complex c1, Complex c2)
        {
            Double denominator = c2.real*c2.real + c2.imag*c2.imag;
            if (denominator > 0)
            {
                Complex numerator = c1 * Conjugate(c2);
                return new Complex(numerator.real / denominator, numerator.imag / denominator);
            }
            else
                throw new ArgumentException("Complex: division by zero");
        }

        /// <summary>
        /// e raised to complex exponent
        /// </summary>
        /// <param name="c1"></param>
        /// <returns>e raised to complex exponent</returns>
        public static Complex Exp(Complex c1)
        {
            return new Complex(Cosh(c1) + Sinh(c1));
        }

        /// <summary>
        /// Natural Log of Complex number
        /// </summary>
        /// <param name="c1"></param>
        /// <returns>Principal value of natural Log of complex number</returns>
        public static Complex Log(Complex c1)
        {
            return new Complex(Math.Log(c1.real*c1.real + c1.imag*c1.imag)/2, ArgRad(c1));
        }


        /// <summary>
        /// Display complex number in real, imag format (electrical engineering notation...)
        /// </summary>
        /// <returns>Sstring representation of the number</returns>
        public override string ToString()
        {
            if (imag >= 0)
                return (String.Format("{0} + j{1}", real, imag));
            else
                return (String.Format("{0} - j{1}", real, -imag));
        }

        // Some trig functions with complex arguments
        
        /// <summary>
        /// sin of complex argument
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex Sin(Complex c1)
        {
            return new Complex(Math.Sin(c1.real) * Math.Cosh(c1.imag), Math.Cosh(c1.real) * Math.Sinh(c1.imag));
        }

        /// <summary>
        /// cos of complex argument
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex Cos(Complex c1)
        {
            return new Complex(Math.Cos(c1.real) * Math.Cosh(c1.imag), -Math.Sin(c1.real) * Math.Sinh(c1.imag));
        }
 
        /// <summary>
        /// tan of complex argument
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex Tan(Complex c1)
        {
            return new Complex(Sin(c1) / Cos(c1));
        }

        /// <summary>
        /// Hyperbolic sin of complex argument
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex Sinh(Complex c1)
        {
            return new Complex(Math.Sinh(c1.real) * Math.Cos(c1.imag),
                Math.Cosh(c1.real) * Math.Sin(c1.imag));
        }

        /// <summary>
        /// Hyperbolic cos of complex argument
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex Cosh(Complex c1)
        {
            return new Complex(Math.Cosh(c1.real) * Math.Cos(c1.imag),
                Math.Sinh(c1.real) * Math.Sin(c1.imag));
        }

        /// <summary>
        /// Hyperbolic tan of complex argument
        /// </summary>
        /// <param name="c1"></param>
        /// <returns></returns>
        public static Complex Tanh(Complex c1)
        {
            return new Complex(Sinh(c1) / Cosh(c1));
        }

        // Override both Object.Equals(obj) and GetHashcode()
        // to permit use of   == and !=   (see C# help)
        public override bool Equals(object obj)
        {
            // Normally object.Equals tests for reference equality.
            // We override to test for real and imag value equality.

            if (obj == null || this.GetType() != obj.GetType())
                return false;
            
            Complex c = (Complex)obj;
            return ((this.real == c.real) && (this.imag == c.imag));
        }

        public override int GetHashCode()
        {
            return (int)real ^ (int)imag;		// bitwise exclusive-or
        }

        /// <summary>
        /// Compare two Complex numbers for equality
        /// </summary>
        /// <param name="c1"></param>
        /// <param name="c2"></param>
        /// <returns>True if both c1 and c2 are equal, or they refer to the same object.</returns>
        public static Boolean operator ==(Complex c1, Complex c2)
        {
            // Object.Equals handles most NULL reference cases before
            // calling Complex.Equals

            return Object.Equals(c1, c2);
        }

        /// <summary>
        /// Compare two Complex numbers for inequality
        /// </summary>
        /// <param name="c1"></param>
        /// <param name="c2"></param>
        /// <returns>True if c1 and c2 are not equal.</returns>
        public static Boolean operator !=(Complex c1, Complex c2)
        {
            return !Object.Equals(c1, c2);
        }
    }
}
