
//
//    Copyright  2010, 2011 Thomas C. McDermott, N5EG
//    This file is part of ABCDmatrix - the 2-Port Network Calculator program.
//
//    ABCDmatrix is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    ABCDmatrix is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with ABCDmatrix, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Text;
using System.Windows.Forms;


namespace ABCDmatrix
{
    /// <summary>
    /// Class that holds a single network element, such as shunt capacitor,
    /// transmission line, etc. Each element is called a tile.
    /// </summary>
    [Serializable]
    public class Tile
    {
        [Description("The type of network element."), Category("Network Type")]
        public NetworkType BlockType { get; set; }  // what type of 2-port network the tile is

        [Description("The characteristics of the network parameters."), Category("Parameter Type")]
        public ParameterType ParamType { get; set; } // whether fixed parameter or frequency-list-per-parameter,

        [Description("The Variable Parameters List."), Category("Frequency-Variable Parameters List")]
        public List<ParameterSet> ParamList { get; set; } // list of the variable parameters (if any).
                                            // It's shown in the browser tab as a collection

        [Description("The value of resistance Ohms in RLC component, or Zo.real for transmission line.\n\rThe value of turns ratio for transformer."),
            Category("Component Parameter")]
        public Double R_Zreal_TrRatio {get; set;} // electrical network component value 1 {R ohms, cable Zo.re} 
        
        [Description("The value of inductance uHy in RLC component & Transformer primary, or VF for transmission line."), Category("Component Parameter")]
        public Double LuHy_VF { get; set; } // electrical network component value 2 {L uHy,  VF}

        [Description("The value of capacitance pF in RLC component, or length in meters for transmission line."), Category("Component Parameter")]
        public Double Cpf_LenMeters { get; set; }  // electrical network component value 3 {C pF, cable length meters}

        [Description("The loss per meter/hz for transmission line, or transformer coupling coefficient (-1..+1)."), Category("Component Parameter")]
        public Double Cablek1Freq_TransfK { get; set; }  // electrical network component value 4 k1 - loss(Freq)

        [Description("The value of Zo.imag for transmission line."), Category("Component Parameter")]
        public Double CableZimag { get; set; } // electrical network component value 5 cable Zo.im

        [Description("Stub termination resistance (real part), Ohms"), Category("Component Parameter")]
        public Double StubTermR { get; set; } // electrical network component value 6 (stub termination R, ohms)

        [Description("Stub termination reactance, Ohms"), Category("Component Parameter")]
        public Double StubTermLuHy { get; set; } // electrical network component value 7 (stub termination L, uHy)

        [Description("Stub termination reactance, Ohms"), Category("Component Parameter")]
        public Double StubTermCpF { get; set; } // electrical network component value 9 (stub termination C, pF)

        [Description("The loss per meter/sqrt(hz) for transmission line"), Category("Component Parameter")]
        public Double Cablek2sqrtFreq { get; set; } // electrical network component value 8 k2 - loss(sqrt(freq))
         

        [Description("Forcing input current condition enabled"), Category("Forcing Current")]
        public Boolean ForceI1I2 { get; set; } // true if I1 should be forced to a value.


        [Description("The name of the file containing the list of parameters vs. frequency."), Category("Frequency-Variable S2P FileName")]
        public String VariableParameterFileName { get; set; } // path+filename for frequency-list of values
                                            // VariableFreq Sparam, Zparam, Yparam, and ABCDparam only

        [Description("The reference resistance for S,Y,Z parameters of the tile."), Category("S,Y,Z Reference Resistance")]
        public Double ReferenceResistance { get; set; }  // Reference resistance for S-Parameters within each tile

        // The following properties can't be browsed directly because they are compound types.
        // Browsable properties are listed below.
        public ParameterSet FixedParam;     // holds fixed 2-port parameters (if any)
        public Complex I1I2ratio;         // ratio of I1 to I2 for forced drive condition
        public Boolean Select;            // true if box has been selected by mouse



        // Define the boundaries of the display.
        private static int LeftMargin = 10;
        private static int RightMargin = 10;
        private static int TopMargin = 10;
        private static int BotMargin = 10;

        // amount wires are inset from tile boundary. Needs to align with BITMAPs of the tiles.
        private static int wireInset = 11;

        // The size of each tile and space between tiles
        private static int cellHeight = 80;
        private static int cellVspace = 32;
        private static int cellWidth = 120;
        private static int cellHspace = 16;

        // Define maximum number of rows and colums on the net design sheet
        private static int maxRow = 4;
        private static int maxCol = 6;


        /// <summary>
        /// Construct a new tile at location, of known network m_type
        /// of standard m_size.
        /// </summary>
        public Tile(NetworkType NetType)
        {
            BlockType = NetType;
            FixedParam = new ParameterSet();
            ParamList = new List<ParameterSet>();
            I1I2ratio = new Complex();
        }

        /// <summary>
        /// default constructor
        /// </summary>
        public Tile()
        {
            BlockType = NetworkType.Empty;   // initialized to empty
            FixedParam = new ParameterSet();
            ParamList = new List<ParameterSet>();
            I1I2ratio = new Complex();
        }

        /// <summary>
        /// Copy constructor
        /// </summary>
        /// <param name="copyfrom"></param>
        public Tile(Tile copyfrom)
        {
            BlockType = copyfrom.BlockType;
            ParamType = copyfrom.ParamType;  
            R_Zreal_TrRatio = copyfrom.R_Zreal_TrRatio;
            LuHy_VF = copyfrom.LuHy_VF;        
            Cpf_LenMeters = copyfrom.Cpf_LenMeters;  
            Cablek1Freq_TransfK = copyfrom.Cablek1Freq_TransfK;    
            CableZimag = copyfrom.CableZimag;     
            StubTermR = copyfrom.StubTermR;  
            StubTermLuHy = copyfrom.StubTermLuHy;  
            Cablek2sqrtFreq = copyfrom.Cablek2sqrtFreq;
            ForceI1I2 = copyfrom.ForceI1I2;
            ReferenceResistance = copyfrom.ReferenceResistance;

            I1I2ratio = new Complex(copyfrom.I1I2ratio); 

            Select = false;   // the cloned tile is not selected even if the parent is selected

            if (copyfrom.VariableParameterFileName != null)
                VariableParameterFileName = String.Copy(copyfrom.VariableParameterFileName);
            else
                VariableParameterFileName = null;

            FixedParam = new ParameterSet(copyfrom.FixedParam);     

            ParamList = new List<ParameterSet>();
            foreach (ParameterSet p in copyfrom.ParamList)
            {
                ParameterSet a = new ParameterSet(p);
                ParamList.Add(a);
            }

         }

        /// <summary>
        /// Draw a tile on the Network design sheet
        /// </summary>
        /// <param name="gr">surface to draw tile</param>
        /// <param name="tileBox">rectangle containing location to draw tile</param>
        public void Draw(Graphics gr, Rectangle tileBox)
        {
            // Draw a tile - the position is determined by the tileBox.
            // The image drawn is dependent on the Network type of the tile,
            // and also depends on the component type for ABCD matrices.

            switch ((int)BlockType)
            {
                case (int)NetworkType.Empty:
                    gr.DrawImage(tilebitmaps.Empty, tileBox);
                    break;

                case (int)NetworkType.ShuntYparRLC:
                case (int)NetworkType.ShuntYserRLC:
                    gr.DrawImage(tilebitmaps.Shunt_Y, tileBox);
                    break;

                case (int)NetworkType.SeriesZparRLC:
                case (int)NetworkType.SeriesZserRLC:
                    gr.DrawImage(tilebitmaps.Series_Z, tileBox);
                    break;

                case (int)NetworkType.SeriesTL:
                    gr.DrawImage(tilebitmaps.Series_TL, tileBox);
                    break;

                case (int)NetworkType.Transformer:
                    gr.DrawImage(tilebitmaps.Transformer, tileBox);
                    break;

                case (int)NetworkType.PiNet:
                    gr.DrawImage(tilebitmaps.Pi_Net, tileBox);
                    break;

                case (int)NetworkType.TeeNet:
                    gr.DrawImage(tilebitmaps.Tee_Net, tileBox);
                    break;

                case (int)NetworkType.ABCDparam:
                    gr.DrawImage(tilebitmaps.ABCDmatrix, tileBox);
                    break;

                case (int)NetworkType.Sparam:
                    gr.DrawImage(tilebitmaps.Smatrix, tileBox);
                    break;

                case (int)NetworkType.Zparam:
                    gr.DrawImage(tilebitmaps.Zmatrix, tileBox);
                    break;

                case (int)NetworkType.Yparam:
                    gr.DrawImage(tilebitmaps.Ymatrix, tileBox);
                    break;

                case (int)NetworkType.ShuntStub:
                    gr.DrawImage(tilebitmaps.Shunt_Stub, tileBox);
                    break;

                case (int)NetworkType.SeriesStub:
                    gr.DrawImage(tilebitmaps.Series_Stub, tileBox);
                    break;

                default:
                    throw new ArgumentException("Tile.Draw: invalid network element type");
            }

            // Outline the tile - heavy box if it's currently selected, normal box if not
            if (Select)
                gr.DrawRectangle(new Pen(Color.Black, 3.0f), tileBox);  // selected box outline
            else
                gr.DrawRectangle(new Pen(Color.Black), tileBox);  // selected box outline
            
        }

        /// <summary>
        /// Draw the bankground grid of the network design page,
        /// including the connector wires between tiles.
        /// </summary>
        /// <param name="gr">Graphics object to draw onto</param>
        public static void BackgroundNetDraw(Graphics gr)
        {
            // 'rect' outlines the bounded display area of the drawable surface.
            // We don't use all of it.

            Rectangle rect = new Rectangle(LeftMargin, TopMargin,
                (int)gr.ClipBounds.Width - RightMargin - LeftMargin,
                (int)gr.ClipBounds.Height - BotMargin - TopMargin);

            Pen grayPen = new Pen(Color.Gray);
            grayPen.DashStyle = DashStyle.Dash;

            Pen bluePen = new Pen(Color.Blue);
            Pen blackPen = new Pen(Color.Black);
            Pen redPen = new Pen(Color.Red);
            Brush nodeBrush = Brushes.Gray;

            int rightBoundary = rect.Left + cellHspace + maxCol * (cellWidth + cellHspace);
            int bottomBoundary = rect.Top + maxRow * (cellHeight + cellVspace);

            Point start = new Point(rect.Left, rect.Top);
            Point stop = new Point(rightBoundary, rect.Top);
            gr.DrawLine(grayPen, start, stop);

            for (int row = 0; row < maxRow; row++)       // draw horizontal light grey lines
            {
                start.Y += cellHeight;
                stop.Y += cellHeight;
                gr.DrawLine(grayPen, start, stop);

                start.Y += cellVspace;
                stop.Y += cellVspace;
                gr.DrawLine(grayPen, start, stop);
            };


            start.X = rect.Left;
            start.Y = rect.Top + wireInset;
            stop.X = rightBoundary;
            stop.Y = rect.Top +  wireInset;

            for (int row = 0; row < maxRow-1; row++)       // draw horizontal blue and red lines
            {
                start.Y += cellHeight;
                stop.Y += cellHeight;
                gr.DrawLine(bluePen, start, stop);

                start.Y += cellVspace - 2 * wireInset;
                stop.Y += cellVspace - 2 * wireInset;
                gr.DrawLine(redPen, start, stop);

                start.Y += 2 * wireInset;
                stop.Y += 2 * wireInset;
            };


            start = new Point(rect.Left, rect.Top);
            stop = new Point(rect.Left, bottomBoundary);
            gr.DrawLine(grayPen, start, stop);

            start.X += cellHspace;
            stop.X += cellHspace;
            gr.DrawLine(grayPen, start, stop);

            for (int col = 0; col < maxCol; col++)       // draw vertical light grey lines
            {
                start.X += cellWidth;
                stop.X += cellWidth;
                gr.DrawLine(grayPen, start, stop);

                start.X += cellHspace;
                stop.X += cellHspace;
                gr.DrawLine(grayPen, start, stop);
            }


            // draw red and blue vertical lines on left side (rows 1 .. maxrow)
            start.X = rect.Left;
            stop.X = rect.Left;
            start.Y = rect.Top - 2 * wireInset;
            stop.Y = start.Y + cellVspace;

            for (int row = 1; row < maxRow; row++)
            {
                start.Y += cellHeight + cellVspace;
                stop.Y += cellHeight + cellVspace;
                start.X++;
                stop.X++;
                gr.DrawLine(bluePen, start, stop);
                start.X--;
                stop.X--;

                start.Y += cellVspace - 2 * wireInset;
                stop.Y += cellHeight - 2 * wireInset;
                start.X--;
                stop.X--;
                gr.DrawLine(redPen, start, stop);
                start.X++;
                stop.X++;
                start.Y -= (cellVspace - 2 * wireInset);
                stop.Y -= (cellHeight - 2 * wireInset);
            }

            // draw red and blue vertical lines on right side (rows 0 .. maxrow-1)
            start.X = rect.Left + cellHspace + maxCol * (cellHspace + cellWidth);
            stop.X = rect.Left + cellHspace + maxCol * (cellHspace + cellWidth);
            start.Y = rect.Top + wireInset;
            stop.Y = start.Y + cellHeight;

            for (int row = 0; row < maxRow-1; row++)
            {
                start.X++;
                stop.X++;
                gr.DrawLine(bluePen, start, stop);
                start.X--;
                stop.X--;

                start.Y += cellHeight - 2 * wireInset;
                stop.Y += cellVspace - 2 * wireInset;
                start.X--;
                stop.X--;
                gr.DrawLine(redPen, start, stop);
                start.X++;
                stop.X++;
                start.Y -= (cellHeight - 2 * wireInset);
                stop.Y -= (cellVspace - 2 * wireInset);
                
                start.Y += cellHeight + cellVspace;
                stop.Y += cellHeight + cellVspace;
            }


            // Show node numbers in the background
            // Draw tile connector wires in background
            
            Point startPoint = new Point(rect.Left, rect.Top + cellHeight/2 - 6);
            StringBuilder nodeNumberString = new StringBuilder();
            Int32 nodeNumber = 0;

            Point UpperWireStart = new Point(rect.Left, rect.Top + wireInset);
            Point UpperWireStop = new Point(rect.Left + cellHspace, rect.Top + wireInset);
            Point LowerWireStart = new Point(rect.Left, rect.Top + cellHeight - wireInset );
            Point LowerWireStop = new Point(rect.Left + cellHspace, rect.Top + cellHeight - wireInset);

            using (Font nodeFont = new Font(FontFamily.GenericSansSerif, 10))
            {

                for (int row = 0; row < maxRow; row++)
                {
                    for (int col = 0; col <= maxCol; col++)
                    {
                        // print node numbers between tiles in light grey

                        nodeNumberString.Append(nodeNumber.ToString());
                        gr.DrawString(nodeNumberString.ToString(), nodeFont, nodeBrush, startPoint);
                        nodeNumber++;
                        nodeNumberString.Length = 0;
                        startPoint.X += cellWidth + cellHspace;

                        // draw red and blue connector wires between tiles

                        gr.DrawLine(bluePen ,UpperWireStart, UpperWireStop);
                        gr.DrawLine(redPen ,LowerWireStart, LowerWireStop);
                        UpperWireStart.X += cellWidth + cellHspace;
                        UpperWireStop.X += cellWidth + cellHspace;
                        LowerWireStart.X += cellWidth + cellHspace;
                        LowerWireStop.X += cellWidth + cellHspace;
                    }

                    nodeNumber--;   // repeat node number on the beginning of the next line

                    startPoint.X = rect.Left;
                    startPoint.Y += cellHeight + cellVspace;

                    UpperWireStart.X = rect.Left;
                    UpperWireStop.X = rect.Left + cellHspace;
                    LowerWireStart.X = rect.Left;
                    LowerWireStop.X = rect.Left + cellHspace;

                    UpperWireStart.Y += cellHeight + cellVspace;
                    UpperWireStop.Y += cellHeight + cellVspace;
                    LowerWireStart.Y += cellHeight + cellVspace;
                    LowerWireStop.Y += cellHeight + cellVspace;
                }
            }
        }

        /// <summary>
        /// Draw each of the network tiles on the network design pane.
        /// </summary>
        /// <param name="gr">Grapics object to draw onto</param>
        /// <param name="tileset">List of tiles to draw</param>
        public static void NetElementDraw(Graphics gr, List<Tile> tileset)
        {
            // 'rect' holds the displayable extent of the page
            //Rectangle rect = new Rectangle(LeftMargin, TopMargin,
            //    (int)gr.ClipBounds.Width - RightMargin - LeftMargin,
            //    (int)gr.ClipBounds.Height - BotMargin - TopMargin);

            // 'tileBox' holds the physical position of each tile as displayed
            Rectangle tileBox = new Rectangle(LeftMargin + cellHspace, TopMargin,
                cellWidth, cellHeight);


            int tileCount = 0;
            foreach (Tile tile in tileset)
            {
                tile.Draw(gr, tileBox);                 // draw the tile at tileBox
                tileBox.X += cellWidth + cellHspace;    // move to the next column
                tileCount++;
                if (tileCount % maxCol == 0)            // display on the next row
                {
                    tileBox.X = LeftMargin + cellHspace;
                    tileBox.Y += cellHeight + cellVspace;
                }
            }
        }

        /// <summary>
        /// Select which tile has been hit by left mouse-click.
        /// Return boolean indicating whether the event changed a tile selection
        /// (which would require refreshing the screen by our parent).
        /// </summary>
        /// <param name="e"></param>
        /// <param name="tileset">List of tiles to check</param>
        public static Boolean HitSelect(MouseEventArgs e, List<Tile> tileset)
        {
            // 'tileBox' holds the physical position of each tile as displayed
            Rectangle tileBox = new Rectangle(LeftMargin + cellHspace, TopMargin,
                cellWidth, cellHeight);

            Boolean tileSelectionChanged = false;       // true if a tile selection changed
            int tileCount = 0;
            foreach (Tile tile in tileset)
            {
                // test if the mouse event is within the tilebox

                if ((e.X >= tileBox.X) &&
                    (e.X <= tileBox.X + tileBox.Width) &&
                    (e.Y >= tileBox.Y) &&
                    (e.Y <= tileBox.Y + tileBox.Height))
                {
                    tile.Select = true;               // select the tile
                    tileSelectionChanged = true;
                }
                else
                {
                    if (tile.Select)
                        tileSelectionChanged = true;    // a tile got deselected
                    tile.Select = false;              // deselect the tile
                }

                tileBox.X += cellWidth + cellHspace;    // move to the next column
                tileCount++;
                if (tileCount % maxCol == 0)            // move to the next row
                {
                    tileBox.X = LeftMargin + cellHspace;
                    tileBox.Y += cellHeight + cellVspace;
                }
            }
            return (tileSelectionChanged);
        }

        /// <summary>
        /// Return the zero-based index of the selected tile, -1 if none selected.
        /// </summary>
        /// <param name="tileset">List of tiles to check</param>
        /// <returns>0-based index of selected tile within the List</returns>
        public static int SelectedIndex(List<Tile> tileset)
        {
            int index = 0;
            foreach (Tile tile in tileset)
            {
                if (tile.Select)
                    return (index);        // return zero-based index of tile that is selected
                else
                    index++;               // else next tile.
            }
            return (-1);                   // no tile selected
        }

        /// <summary>
        /// Return tile (rather than index) that's been selected
        /// </summary>
        /// <param name="tileset"></param>
        /// <returns> the tile object that is selected within the List</returns>
        public static Tile SelectedTile(List<Tile> tileset)
        {
            foreach (Tile tile in tileset)
            {
                if (tile.Select)
                    return (tile);        // return tile that is selected
            }
            return (null);                // no tile selected
        }

        /// <summary>
        /// Read the S2P file into the tile.
        /// Parse the file, make sure it's the same type as the file,
        /// and build the list of frequencies and parameter values vs. frequency.
        /// </summary>
        /// <param name="tile">The tile into which to insert parameter list.</param>
        /// <param name="fileSpec">name of the file to read S2P from.</param>
        /// <returns>true if sucessful, false if error reading the file.</returns>
        public static Boolean ReadS2PFile(Tile tile, string fileSpec)
        {
            // Touchstone 1.1 Specification (EIA/IBIS Open Forum, 2002). 
            // S2P files use # to indicate the file format line, and ! for comments.
            // Format is:  # <frequency unit> <parameter> <format> R <n>
            // <frequency unit> is GHz, MHz, KHz, or Hz. Default is GHz.
            // <parameter> is S (S parameter), Z, Y, H, or G. Default is S.
            // <format> is DB (dB-angle), MA (magnitude-angle), or RI (real-imaginary).
            //     Angles are always in degrees.
            // <n> is the reference resistance in ohms. Default is 50 ohms.
            // Note: There is no format specifier for ABCD networks.

            String inputline = null;
            String[] separator = { " ", "\t" };    // separators for tokenizing a line
            String[] token;                        // tokenized fields from a line
            Int32 FreqMultiplier = 0;
            NetworkType nettype = NetworkType.Empty;
            String format = null;

            
            using (StreamReader sr = new StreamReader(fileSpec, Encoding.ASCII))
            {
                while (!sr.EndOfStream)
                {
                    inputline = sr.ReadLine();
                    inputline = inputline.Trim();           // strip whitespace from line
                    if (inputline.StartsWith("#"))          // read until we get the format line
                        break;
                }

                token = inputline.Split(separator, StringSplitOptions.RemoveEmptyEntries);  // tokenize format line

                // Frequency units
                if (token[1].Equals("GHz", StringComparison.CurrentCultureIgnoreCase)) FreqMultiplier = 1000000000;
                if (token[1].Equals("MHz", StringComparison.CurrentCultureIgnoreCase)) FreqMultiplier = 1000000;
                if (token[1].Equals("KHz", StringComparison.CurrentCultureIgnoreCase)) FreqMultiplier = 1000;
                if (token[1].Equals("Hz", StringComparison.CurrentCultureIgnoreCase)) FreqMultiplier = 1;

                // S2P file type
                if (token[2].Equals("S", StringComparison.CurrentCultureIgnoreCase)) nettype = NetworkType.Sparam;
                if (token[2].Equals("Z", StringComparison.CurrentCultureIgnoreCase)) nettype = NetworkType.Zparam;
                if (token[2].Equals("Y", StringComparison.CurrentCultureIgnoreCase)) nettype = NetworkType.Yparam;

                if (nettype != tile.BlockType)
                {
                    StringBuilder errorMessage = new StringBuilder();
                    errorMessage.AppendFormat("Error - tile type does not match S2P file type.\n\r" +
                        "Tile type:\t\t\t{0} \n\rS2P file format character:\t{1}", tile.BlockType.ToString(), token[2]);

                    MessageBox.Show(errorMessage.ToString(), "Format mismatch",
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return false;
                }

                // How the data is formatted (DB, RI, or MA)
                format = token[3].ToUpper();

                // Reference resistance for S parameters
                tile.ReferenceResistance = Convert.ToDouble(token[5]);

                // Read each data line from the S2P file, ignore comment lines

                inputline = sr.ReadLine();                  // get next line
                while (!sr.EndOfStream)
                {
                    inputline = inputline.Trim();           // strip whitespace from line
                    if (!inputline.StartsWith("!"))         // parse line if it's not a comment
                    {
                        token = inputline.Split(separator, StringSplitOptions.RemoveEmptyEntries);

                        ParameterSet line = new ParameterSet();
                        double mag, angle;

                        try
                        {
                            line.frequency = Convert.ToDouble(token[0]) * FreqMultiplier;

                            // read in 8 numbers (4 parameters) based on the format

                            if (String.Compare(format, "DB", false) == 0)
                            {
                                // convert mag from dB to linear
                                mag = Math.Pow(10, Convert.ToDouble(token[1]) / 20);
                                // convert degrees to radians
                                angle = Math.PI / 180 * Convert.ToDouble(token[2]);
                                line.P11A.real = mag * Math.Cos(angle);
                                line.P11A.imag = mag * Math.Sin(angle);

                                mag = Math.Pow(10, Convert.ToDouble(token[3]) / 20);
                                angle = Math.PI / 180 * Convert.ToDouble(token[4]);
                                line.P21C.real = mag * Math.Cos(angle);
                                line.P21C.imag = mag * Math.Sin(angle);

                                mag = Math.Pow(10, Convert.ToDouble(token[5]) / 20);
                                angle = Math.PI / 180 * Convert.ToDouble(token[6]);
                                line.P12B.real = mag * Math.Cos(angle);
                                line.P12B.imag = mag * Math.Sin(angle);

                                mag = Math.Pow(10, Convert.ToDouble(token[7]) / 20);
                                angle = Math.PI / 180 * Convert.ToDouble(token[8]);
                                line.P22D.real = mag * Math.Cos(angle);
                                line.P22D.imag = mag * Math.Sin(angle);
                            }

                            if (String.Compare(format, "MA", false) == 0)
                            {
                                mag = Convert.ToDouble(token[1]);
                                // convert degrees to radians
                                angle = Math.PI / 180 * Convert.ToDouble(token[2]);
                                line.P11A.real = mag * Math.Cos(angle);
                                line.P11A.imag = mag * Math.Sin(angle);

                                mag = Convert.ToDouble(token[3]);
                                angle = Math.PI / 180 * Convert.ToDouble(token[4]);
                                line.P21C.real = mag * Math.Cos(angle);
                                line.P21C.imag = mag * Math.Sin(angle);

                                mag = Convert.ToDouble(token[5]);
                                angle = Math.PI / 180 * Convert.ToDouble(token[6]);
                                line.P12B.real = mag * Math.Cos(angle);
                                line.P12B.imag = mag * Math.Sin(angle);

                                mag = Convert.ToDouble(token[7]);
                                angle = Math.PI / 180 * Convert.ToDouble(token[8]);
                                line.P22D.real = mag * Math.Cos(angle);
                                line.P22D.imag = mag * Math.Sin(angle);
                            }

                            if (String.Compare(format, "RI", false) == 0)
                            {
                                line.P11A.real = Convert.ToDouble(token[1]);
                                line.P11A.imag = Convert.ToDouble(token[2]);
                                line.P21C.real = Convert.ToDouble(token[3]);
                                line.P21C.imag = Convert.ToDouble(token[4]);
                                line.P12B.real = Convert.ToDouble(token[5]);
                                line.P12B.imag = Convert.ToDouble(token[6]);
                                line.P22D.real = Convert.ToDouble(token[7]);
                                line.P22D.imag = Convert.ToDouble(token[8]);
                            }
                        }
                        catch (FormatException /*f*/)
                        {
                            StringBuilder errorMessage = new StringBuilder("Input line from s2p file: \n\r");

                            for (int i = 0; i < 9; i++)
                                errorMessage.AppendFormat("   parameter {0}:\t {1}\n\r", i, token[i]);

                            MessageBox.Show(errorMessage.ToString(), "Data format error in s2p file - cannot continue",
                                MessageBoxButtons.OK, MessageBoxIcon.Error);
                            return false;
                        }

                        tile.ParamList.Add(line);               // add the parameters to the list
                    }
                    inputline = sr.ReadLine();               // read next line
                }
                return true;                                // success
            }
        }

        /// <summary>
        /// Extract ABCD parameters from tile at Frequency, return as ParameterSet.
        /// </summary>
        /// <param name="Frequency">Frequency </param>
        /// <param name="LastTile">False = use all 4 2-port parameters (normal use).
        /// True = this is the last (terminating tile) of the network, and ABCD will be
        /// derived using only S11, Z11, or Y11, all other 2-port parameters will be neglected.</param>
        /// <returns>ParameterSet containing ABCD parameters at specified Frequency.</returns>
        public ParameterSet ExtractABCD(Double Frequency, Boolean LastTile) // extract ABCD at Frequency from Tile
        {

            Double XL, XC, reactance, resistance;     // store interim calculations
            Double conductance, susceptance, Lsusceptance, Csusceptance;
            Complex Zstubload = new Complex();
            Complex impedance = new Complex();
            Complex admittance = new Complex();
            Complex Z0 = new Complex();
            Complex A = new Complex();
            Complex B = new Complex();
            Complex C = new Complex();
            Complex D = new Complex();

            // Extract the ABCD paramters from the tile.
            // S, Z, and Y need to be adjusted by the tile's reference resistance.
            // Fixed types directly convert the parameters.
            // Variable types must interpolate between the specified frequencies then extract.
            // ABCD tile specification is not normalized to any reference resistance.

            try
            {
                Z0.real = ReferenceResistance;     // get reference impedance for this tile from tile spec
                Z0.imag = 0;

                ParameterSet r;

                switch (BlockType)      // find type of tile and derive ABCD for that tile
                {
                    case NetworkType.ABCDparam:
                        if (ParamType == ParameterType.Fixed)  // extract ABCD from 4 fixed-frequency parameters
                            r = FixedParam;
                        else                 // extract ABCD(f) by interpolation from nearest Frequencies
                            r = FindInterpolate(Frequency);

                        A = r.P11A;
                        B = r.P12B;
                        C = r.P21C;
                        D = r.P22D;
                        
                        break;

                    case NetworkType.Zparam:
                        if (ParamType == ParameterType.Fixed)   // extract ABCD from 4 fixed-frequency parameters
                            r = FixedParam;
                        else                 // extract ABCD(f) by interpolation from nearest Frequencies
                            r = FindInterpolate(Frequency);

                        r = Z0*r;             // Convert normalized Z-parameters into ohms

                        if (!LastTile /*|| m_ForceI1*/) // use full set if not the last tile, or forcing condition requested
                        {
                            Complex Zmag = new Complex();
                            Zmag = r.P11A * r.P22D - r.P12B * r.P21C;
                            A = r.P11A / r.P21C;
                            B = Zmag / r.P21C;
                            C = Complex.One / r.P21C;
                            D = r.P22D / r.P21C;
                        }
                        else                    // last tile can only terminate the network
                        {
                            A = Complex.One;
                            B = Complex.Zero;
                            C = Complex.One / r.P11A;
                            D = Complex.One;
                        }
                        
                        break;

                    case NetworkType.Yparam:
                        if (ParamType == ParameterType.Fixed)  // extract ABCD from 4 fixed-frequency parameters
                            r = FixedParam;
                        else                 // extract ABCD(f) by interpolation from nearest Frequencies
                            r = FindInterpolate(Frequency);

                        r = r/Z0;             // Convert normalized Y-parameters into mhos.

                        if (!LastTile /*|| m_ForceI1*/)
                        {
                            Complex Ymag = new Complex();
                            Ymag = r.P11A * r.P22D - r.P12B * r.P21C;
                            A = -r.P22D / r.P21C;
                            B = -Complex.One / r.P21C;
                            C = -Ymag / r.P21C;
                            D = -r.P11A / r.P21C;
                        }
                        else                    // last tile can only terminate the network
                        {
                            A = Complex.One;
                            B = Complex.Zero;
                            C = r.P11A;
                            D = Complex.One;
                        }

                        break;

                    case NetworkType.Sparam:
                        if (ParamType == ParameterType.Fixed)  // extract ABCD from 4 fixed-frequency parameters
                            r = FixedParam;
                        else                 // extract ABCD(f) by interpolation from nearest Frequencies
                            r = FindInterpolate(Frequency);

                        if (!LastTile /*|| m_ForceI1*/)
                        {
                            A = ((Complex.One + r.P11A) * (Complex.One - r.P22D) +
                                r.P12B * r.P21C) / (Complex.Two * r.P21C);
                            B = Z0 * (((Complex.One + r.P11A) * (Complex.One + r.P22D) -
                                r.P12B * r.P21C) / (Complex.Two * r.P21C));
                            C = (((Complex.One - r.P11A) * (Complex.One - r.P22D) -
                                r.P12B * r.P21C) / (Complex.Two * r.P21C)) / Z0;
                            D = ((Complex.One - r.P11A) * (Complex.One + r.P22D) +
                                r.P12B * r.P21C) / (Complex.Two * r.P21C);
                        }
                        else                    // last tile can only terminate the network
                        {
                            A = Complex.One;
                            B = Complex.Zero;
                            C = Complex.One / (new Complex(ReferenceResistance) * (Complex.One + r.P11A) /
                                (Complex.One - r.P11A));
                            D = Complex.One;
                        }
                        break;

                    case NetworkType.SeriesTL:
                    case NetworkType.ShuntStub:
                    case NetworkType.SeriesStub:

                        // Utilize the low-cable loss approximation 
   
                        Complex Zcable = new Complex(R_Zreal_TrRatio, CableZimag);
                        Complex Ycable = Complex.One / Zcable;

                        // alpha is the loss in nepers
                        // 1 neper = 10 log (e^2) = 8.686 dB
                        Double alphal = CableSet.LossDB(Frequency, Cpf_LenMeters, Cablek1Freq_TransfK,
                            Cablek2sqrtFreq);

                        double NeperdB = 10 * Math.Log10(Math.E*Math.E);
                        alphal = alphal / NeperdB;

                        // beta is defined in terms of radians of a wavelength.
                        // Need to convert length in meters into fractional wavelength at f considering
                        // the velocity factor.
                        // cable radians = 2*Pi*[ (len / lambda) /VF]
                        // cable beta = 2 PI f / VF
                        // lambda * f = c, thus lambda = c/f.   Then len/lambda = len*f/c

                        Double betal = 2 * Math.PI * Frequency * Cpf_LenMeters / (299792458 * LuHy_VF);

                        // gamma * length is the propagation constant over the whole length of the cable
                        // gammal = alphal + j betal. 
                        Complex gammal = new Complex(alphal, betal);

                        // For lossless line, A = cos(beta*len), B = jZo sin(beta*len),
                        //     C = jY0 sin(beta*len), D = cos(beta*len)
                        // For lossy line, A = cosh(gamma*len), B = Zo sinh(gamma*len),
                        //     C = Y0 sinh(gamma*len), D = cosh(gamma*len)

                        // C# library does not have complex cosh() and sinh() functions.
                        // These have been added to the Complex Class in this software.
                        // Use the expansions, with z = x+jy :  
                        //   cosh(z) = cosh(x)*cos(y) + j sinh(x)*sin(y)
                        //   sinh(z) = sinh(x)*cos(y) + j cosh(x)*sin(y)
                        //   tanh(z) = sinh(z)/cosh(z)
                        //

                        if (BlockType == NetworkType.SeriesTL)
                        {
                            A = Complex.Cosh(gammal);
                            B = Zcable * Complex.Sinh(gammal);
                            C = Ycable * Complex.Sinh(gammal);
                            D = A;
                        }
                        
                        if (BlockType == NetworkType.ShuntStub)
                        {
                            Zstubload = ComputeStubTermZ(StubTermR, StubTermLuHy, StubTermCpF, Frequency);

                            // Compute stub input impedance for stub terminated in complex load impedance
                            Complex StubZ = Zcable * (Zstubload + Zcable * Complex.Tanh(gammal)) /
                                (Zcable + Zstubload * Complex.Tanh(gammal));

                            A = Complex.One;
                            B = Complex.Zero;
                            C = Complex.One / StubZ;  // note shunt stub uses Ystub
                            D = Complex.One;
                        }

                        if (BlockType == NetworkType.SeriesStub)
                        {
                            Zstubload = ComputeStubTermZ(StubTermR, StubTermLuHy, StubTermCpF, Frequency);

                            // Compute stub input impedance for stub terminated in complex load impedance
                            Complex StubZ = Zcable * (Zstubload + Zcable * Complex.Tanh(gammal)) /
                                (Zcable + Zstubload * Complex.Tanh(gammal));

                            A = Complex.One;
                            B = StubZ;           // series stub uses Zstub
                            C = Complex.Zero;
                            D = Complex.One;
                        }
                        break;

                    case NetworkType.SeriesZparRLC:

                        // R is in ohms. 
                        resistance = R_Zreal_TrRatio;
                        if (resistance != 0)
                            conductance = 1 / resistance;
                        else
                            conductance = 0; // R=0 is shorthand for no resistor.

                        // L is in uHy. 
                        if (LuHy_VF != 0)
                            Lsusceptance = -1 / (2 * Math.PI * Frequency * LuHy_VF / 1e6);
                        else
                            Lsusceptance = 0; // L=0 is shorthand for no inductor.

                        // C is in pF.
                        Csusceptance = (2 * Math.PI * Frequency * Cpf_LenMeters / 1E12);

                        susceptance = Csusceptance + Lsusceptance; 

                        admittance.real = conductance;
                        admittance.imag = susceptance;

                        impedance = Complex.One / admittance;

                        A = Complex.One;
                        B = impedance;
                        C = Complex.Zero;
                        D = Complex.One;
                        break;

                    case NetworkType.SeriesZserRLC:

                        resistance = R_Zreal_TrRatio; // R ohms

                        XL = 2 * Math.PI * Frequency * LuHy_VF / 1e6;   // L uHy

                        // C is in pF.   
                        if (Cpf_LenMeters != 0)
                            XC = -1 / (2 * Math.PI * Frequency * Cpf_LenMeters / 1E12);
                        else
                            XC = 0; // C=0 is shorthand for no capacitor.

                        reactance = XL + XC;

                        A = Complex.One;
                        B = new Complex(resistance, reactance);
                        C = Complex.Zero;
                        D = Complex.One;
                        break;

                    case NetworkType.ShuntYparRLC:

                        // R is in ohms. 
                        resistance = R_Zreal_TrRatio;
                        if (resistance != 0)
                            conductance = 1 / resistance;
                        else
                            conductance = 0;    // R=0 is shorthand for no resistor.

                        // L is in uHy. 
                        if (LuHy_VF != 0)
                            Lsusceptance = -1 / (2 * Math.PI * Frequency * LuHy_VF / 1e6);
                        else
                            Lsusceptance = 0;   // L=0 is shorthand for no inductor.

                        // C is in pF.
                        Csusceptance = (2 * Math.PI * Frequency * Cpf_LenMeters / 1E12);

                        susceptance = Lsusceptance + Csusceptance; 

                        admittance.real = conductance;
                        admittance.imag = susceptance;

                        A = Complex.One;
                        B = Complex.Zero;
                        C = admittance;
                        D = Complex.One;
                        break;

                    case NetworkType.ShuntYserRLC:

                        resistance = R_Zreal_TrRatio; // R ohms

                        XL = 2 * Math.PI * Frequency * LuHy_VF / 1e6;   // L uHy

                        // C is in pF.   
                        if (Cpf_LenMeters != 0)
                            XC = -1 / (2 * Math.PI * Frequency * Cpf_LenMeters / 1E12);
                        else
                            XC = 0; // C=0 is shorthand for no capacitor.

                        reactance = XL + XC;

                        impedance.real = resistance;
                        impedance.imag = reactance;

                        admittance = Complex.One / impedance;

                        A = Complex.One;
                        B = Complex.Zero;
                        C = admittance;
                        D = Complex.One;
                        break;

                    case NetworkType.Transformer:     // transformer with specified coupling coefficient and primary inductance

                        Double LMutual = Cablek1Freq_TransfK * R_Zreal_TrRatio * LuHy_VF;
                        Double LPrimary = LuHy_VF;
                        Double LSecondary = LuHy_VF * R_Zreal_TrRatio * R_Zreal_TrRatio;
                        Double Omega = 2.0 * Math.PI * Frequency;
                        
                        Complex Z1 = new Complex(0, Omega * 10E-6 * (LPrimary - LMutual));    // jw(Lp-M)
                        Complex Z2 = new Complex(0, Omega * 10E-6 * (LSecondary - LMutual));  // jw(Ls-M)
                        Complex Z3 = new Complex(0, Omega * 10E-6 * LMutual);                 // jwM
                        
                        A = Complex.One + (Z1 / Z3);
                        B = Z1 + Z2 + ((Z1 * Z2) / Z3);
                        C = Complex.One / Z3;
                        D = Complex.One + (Z2 / Z3);

                        break;

                    case NetworkType.Empty:     // this provides just connection function (matrix = unity).

                        A = Complex.One;
                        B = Complex.Zero;
                        C = Complex.Zero;
                        D = Complex.One;
                        break;

                    default:
                        throw new ArgumentException("Analyze: Invalid tile network type");
                }

                ParameterSet result = new ParameterSet();
                result.frequency = Frequency;
                result.P11A = A;
                result.P12B = B;
                result.P21C = C;
                result.P22D = D;
                return result;
            }
            catch (ArgumentException a)
            {
                MessageBox.Show("Error calculating concatenated matrix transfer function: \n\r", a.Message,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);

                ParameterSet result = new ParameterSet();
                result.frequency = -1;            // signal an error to caller
                result.P11A = Complex.One;        //return identity matrix to ease debugging
                result.P12B = Complex.Zero;
                result.P21C = Complex.Zero;
                result.P22D = Complex.One;
                return result;
            }
        }

        /// <summary>
        /// Compute the impedance of a stub termination at a given Frequency. The termination network is 
        /// C in parallel with R+L
        /// </summary>
        /// <param name="stubTermR">series resistance portion, ohms</param>
        /// <param name="stubTermLuHy">series inducatance portion, uHy</param>
        /// <param name="stubTermCpF">parallel capacitance, pF.</param>
        /// <returns>Stub Termination Impedance (complex)</returns>
        public static Complex ComputeStubTermZ(double stubTermR, double stubTermLuHy, double stubTermCpF, double frequency)
        {                            
            // compute stub load impedance, Rs+Ls || Cp at this frequency 
            
            if (stubTermR == 0.0 && stubTermLuHy == 0.0)    // stub terminated in dead short
                return Complex.Zero;

            double XL = 2.0 * Math.PI * frequency * stubTermLuHy * 10e-6;

            Complex impedance = new Complex(stubTermR, XL);

            if (stubTermCpF == 0)                       // if no capacitor present
                return impedance;

            Complex admittance = Complex.One / impedance;
            double Csusceptance = -2.0 * Math.PI * frequency * stubTermCpF * 10e-12;
            admittance += new Complex(0, Csusceptance);

            return Complex.One / admittance;
        }

        /// <summary>
        /// Find the bracketing ParameterSets by Frequency then compute interpolated ParameterSet value
        /// at Frequency. The ParameterSets in the collection must be ordered by increasing frequency.
        /// </summary>
        /// <param name="Frequency">Frequency at which to interpolate</param>
        /// <returns>Interpolated ParameterSet</returns>
        private ParameterSet FindInterpolate(Double Frequency)
        {
            ParameterSet r;
            // Assumes that the frequencies in m_paramList are sorted in ascending order

            ParameterSet previous = ParamList[0];                  // first ParameterSet
            ParameterSet next = ParamList[ParamList.Count - 1];  // last ParameterSet

            foreach (ParameterSet ps in ParamList) // find two parameter sets that bracket Frequency
            {
                if (ps.frequency == Frequency)        // found an exact match
                {
                    previous = ps;              // previous and next are sets that bracket the
                    next = ps;                  // frequency point
                    break;
                }

                if (ps.frequency < Frequency)
                    previous = ps;

                if (ps.frequency > Frequency)
                {
                    next = ps;
                    break;
                }
            }
            // Interpolation requires Real/Imaginary format to prevent phase wraparound,
            // the parameters read from S2P file have already been converted to this format.

            r = ParameterSet.Interpolate(previous, next, Frequency);
            return r;
        }

        [Description("The real part of matrix element 11 in fixed parameterized 2-port."), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam11real
        {
            get
            {
                return FixedParam.P11A.real;
            }
            set
            {
                FixedParam.P11A.real = value;
            }
        }

        [Description("The imaginary part of matrix element 11 in fixed parameterized 2-port."), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam11imag
        {
            get
            {
                return FixedParam.P11A.imag;
            }
            set
            {
                FixedParam.P11A.imag = value;
            }
        }

        [Description("The real part of matrix element 21 in fixed parameterized 2-port."), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam21real
        {
            get
            {
                return FixedParam.P21C.real;
            }
            set
            {
                FixedParam.P21C.real = value;
            }
        }

        [Description("The imaginary part of matrix element 21 in fixed parameterized 2-port"), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam21imag
        {
            get
            {
                return FixedParam.P21C.imag;
            }
            set
            {
                FixedParam.P21C.imag = value;
            }
        }

        [Description("The real part of matrix element 22 in fixed parameterized 2-port"), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam22real
        {
            get
            {
                return FixedParam.P22D.real;
            }
            set
            {
                FixedParam.P22D.real = value;
            }
        }

        [Description("The imaginary part of matrix element 22 in fixed parameterized 2-port"), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam22imag
        {
            get
            {
                return FixedParam.P22D.imag;
            }
            set
            {
                FixedParam.P22D.imag = value;
            }
        }

        [Description("The real part of matrix element 12 in fixed parameterized 2-port"), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam12real
        {
            get
            {
                return FixedParam.P12B.real;
            }
            set
            {
                FixedParam.P12B.real = value;
            }
        }

        [Description("The imaginary part of matrix element 12 in fixed parameterized 2-port"), Category("2-port Matrix Fixed Parameter")]
        public double FixedParam12imag
        {
            get
            {
                return FixedParam.P12B.imag;
            }
            set
            {
                FixedParam.P12B.imag = value;
            }
        }

        [Description("The real part of forcing current I1 (left side) divided by current I2 (right side)"), Category("Forcing Current")]
        public double ForceI1I2real
        {
            get
            {
                return I1I2ratio.real;
            }
            set
            {
                I1I2ratio.real = value;
            }
        }

        [Description("The imaginary part of forcing current I1 (left side) divided by current I2 (right side)"), Category("Forcing Current")]
        public double ForceI1I2imag
        {
            get
            {
                return I1I2ratio.imag;
            }
            set
            {
                I1I2ratio.imag = value;
            }
        }
   
    }
    

    /// <summary>
    /// Type of networks that can be represented by a 2-port
    /// </summary>
    public enum NetworkType
    {
        /// <summary>
        /// Empty type. May throw an exception if used.
        /// </summary>
        Empty = 0,      
        /// <summary>
        /// ABCD-parameter network
        /// </summary>
        ABCDparam,      
        /// <summary>
        /// S-parameter network
        /// </summary>
        Sparam,         
        /// <summary>
        /// Z-parameter network
        /// </summary>
        Zparam, 
        /// <summary>
        /// Y-parameter network
        /// </summary>
        Yparam,         
        /// <summary>
        /// ABCD shunt admittance using series RLC network
        /// </summary>
        ShuntYserRLC,   
        /// <summary>
        /// ABCD shunt admittance using parallel RLC
        /// </summary>
        ShuntYparRLC,   
        /// <summary>
        /// ABCD series impedance using series RLC
        /// </summary>
        SeriesZserRLC,  
        /// <summary>
        /// ABCD series impedance using parallel RLC
        /// </summary>
        SeriesZparRLC,  
        /// <summary>
        /// Series transmission line
        /// </summary>
        SeriesTL,       
        /// <summary>
        /// Step up/down ideal transformer
        /// </summary>
        Transformer,    
        /// <summary>
        /// Shunt stub with termination (Rs, Ls, Cp)
        /// </summary>
        ShuntStub,      
        /// <summary>
        /// Series stub with termination (Rs, Ls, Cp)
        /// </summary>
        SeriesStub,     
        /// <summary>
        /// Pi network - not implemented
        /// </summary>
        PiNet,          
        /// <summary>
        /// T network - not implemented
        /// </summary>
        TeeNet          
   }


    /// <summary>
    /// How the parameters are defined for the tile
    /// </summary>
    public enum ParameterType
    {
        /// <summary>
        /// Parameters are fixed complex number with no freqnecy dependence
        /// </summary>
        Fixed = 0,      
        /// <summary>
        /// Parameters are a frequency-dependent list of complex numbers
        /// </summary>
        VariableFreq     
    }


}
