/*******************************************************************************
* File Name: VDACQ.c  
* Version 1.50
*
*  Description:
*    This file provides the source code to the API for the 8-bit Voltage DAC 
*    (VDAC8) User Module.
*
*   Note:
*     Any unusual or non-standard behavior should be noted here. Other-
*     wise, this section should remain blank.
*
*******************************************************************************
* Copyright 2008-2010, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
********************************************************************************/

#include "cytypes.h"
#include "VDACQ.h"

uint8 VDACQ_initVar = 0;


/*******************************************************************************
* Function Name: VDACQ_Init
********************************************************************************
* Summary:
*     Initialize to the schematic state
* 
* Parameters:  
*  void:  
*
* Return: 
*  (void)
*
* Theory: 
*
* Side Effects: 
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACQ_Init(void) 
{
    VDACQ_CR0 = (VDACQ_MODE_V )  ;    

    /* Set default data source */
    if(VDACQ_DEFAULT_DATA_SRC != 0 )    
    {
        VDACQ_CR1 = (VDACQ_DEFAULT_CNTL | VDACQ_DACBUS_ENABLE) ;   
    }
    else
    {
        VDACQ_CR1 = (VDACQ_DEFAULT_CNTL | VDACQ_DACBUS_DISABLE) ;  
    } 
    
    /* Set default strobe mode */
    if(VDACQ_DEFAULT_STRB != 0)
    {
        VDACQ_Strobe |= VDACQ_STRB_EN ;
    }
    
    /* Set default range */
    VDACQ_SetRange(VDACQ_DEFAULT_RANGE); 
    
    /* Set default speed */
    VDACQ_SetSpeed(VDACQ_DEFAULT_SPEED);
}


/*******************************************************************************
* Function Name: VDACQ_Enable
********************************************************************************
* Summary:
*     Enable the VDAC8
* 
* Parameters:  
*  void:  
*
* Return: 
*  (void)
*
* Theory: 
*
* Side Effects: 
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACQ_Enable(void) 
{
    VDACQ_PWRMGR |= VDACQ_ACT_PWR_EN;
    VDACQ_STBY_PWRMGR |= VDACQ_STBY_PWR_EN;
}


/*******************************************************************************
* Function Name: VDACQ_Start
********************************************************************************
*
* Summary:
*  The start function initializes the voltage DAC with the default values, 
*  and sets the power to the given level.  A power level of 0, is the same as executing
*  the stop function.
*
* Parameters:  
*  Power:   Sets power level between off (0) and (3) high power
*
* Return: 
*  (void) 
*
* Global variables:
*  VDACQ_initVar: Is modified when this function is called for the first 
*   time. Is used to ensure that initialization happens only once.
*
* Reenrancy: 
*  No
*
*******************************************************************************/
void VDACQ_Start(void) 
{  
    /* Hardware initiazation only needs to occure the first time */
    if ( VDACQ_initVar == 0)  
    { 
        VDACQ_Init();
       
        VDACQ_initVar = 1;
    }  
   
    /* Enable power to DAC */
    VDACQ_Enable();
    
    /* Set default value */
    VDACQ_SetValue(VDACQ_DEFAULT_DATA); 
}


/*******************************************************************************
* Function Name: VDACQ_Stop
********************************************************************************
*
* Summary:
*  Powers down DAC to lowest power state.
*
* Parameters:  
*  (void)  
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACQ_Stop(void) 
{
   /* Disble power to DAC */
   VDACQ_PWRMGR &= ~VDACQ_ACT_PWR_EN;
   VDACQ_STBY_PWRMGR &= ~VDACQ_STBY_PWR_EN;
}


/*******************************************************************************
* Function Name: VDACQ_SetSpeed
********************************************************************************
*
* Summary:
*  Set DAC speed
*
* Parameters:  
*  power:   Sets speed value 
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACQ_SetSpeed(uint8 speed) 
{
   /* Clear power mask then write in new value */
   VDACQ_CR0 &= ~VDACQ_HS_MASK ;    
   VDACQ_CR0 |=  ( speed & VDACQ_HS_MASK) ;    
}


/*******************************************************************************
* Function Name: VDACQ_SetRange
********************************************************************************
*
* Summary:
*  Set one of three current ranges.
*
* Parameters:  
*  Range:  Sets one of Three valid ranges.
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACQ_SetRange(uint8 range) 
{
   VDACQ_CR0 &= ~VDACQ_RANGE_MASK ;      /* Clear existing mode */
   VDACQ_CR0 |= ( range & VDACQ_RANGE_MASK ) ; /*  Set Range  */
   VDACQ_DacTrim();
}


/*******************************************************************************
* Function Name: VDACQ_SetValue
********************************************************************************
*
* Summary:
*  Set 8-bit DAC value
*
* Parameters:  
*  value:  Sets DAC value between 0 and 255.
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*******************************************************************************/
void VDACQ_SetValue(uint8 value) 
{
   VDACQ_Data = value;                             /*  Set Value  */
   
   /* PSOC3 silicons earlier to ES3 require a double write */
   #if (VDACQ_PSOC3_ES2 || VDACQ_PSOC5_ES1)
       VDACQ_Data = value;
   #endif
}


/*******************************************************************************
* Function Name: VDACQ_DacTrim
********************************************************************************
*
* Summary:
*  Set the trim value for the given range.
*
* Parameters:  
*   range:  1V or 4V range.  See constants.  
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACQ_DacTrim(void) 
{
    uint8 mode;
    
    mode = ((VDACQ_CR0 & VDACQ_RANGE_MASK) >> 2) + VDACQ_TRIM_M7_1V_RNG_OFFSET;
    
    VDACQ_TR = CY_GET_XTND_REG8((uint8 *)(VDACQ_DAC_TRIM_BASE + mode));
}


/* [] END OF FILE */
