/*	DSK.C -- Routines to communicatw with applications running
 *	on the TI DSP Starter Kit
 *
 *	J. Bloom, 1/2/95
 *
 *	Copyright (c) 1995, American Radio Relay League
 *	This program can be used free of charge for noncommercial use.
 *	Contact the American Radio Relay League, Inc, 225 Main Street,
 *	Newington, CT 06111 (tel: 203-666-1541 x276) for information
 *	regarding commercial use.
 */

#include <stdio.h>
#include <stdlib.h>
#include <bios.h>
#include <dos.h>

#include "comport.h"
#include "dsk.h"

/*	r e c v _ d s p
 *
 *	Receive a character from the DSP. Return -1 after "timeout" msec
 *	of waiting for a char.
 */
int
recv_dsp(COMM *comm, unsigned timeout)
{
	int c;
	long tval = timeout;

	while ((c = com_in(comm)) == -1) {
		if (tval <= 0) {
			return -1;
		}
		if (bioskey(1) != 0)
			if ((bioskey(0) & 0xff) == 0x1b)
				return -1;
		delay(10);
		tval -= 10;
	}
	return c;
}

/*	r e c v _ d s p _ 1 6
 *
 */
long
recv_dsp_16(COMM *comm, unsigned timeout)
{
	long l, h;

	if ((l = recv_dsp(comm, timeout)) == -1)
		return -1;
	if ((h = recv_dsp(comm, 100)) == -1)
		return -1;
	return (h << 8) | l;
}

/*	s e n d _ d s p
 *
 *	Send a byte to the DDSK
 */
int
send_dsp(COMM *comm, int dat)
{
	return send_com(comm, dat & 0xff, 0);
}

/*	s e n d _ d s p _ 1 6
 *
 *	Send a 16-bit value to the DSK
 */
int
send_dsp_16(COMM *comm, int dat)
{
	if (send_dsp(comm, dat) != 0)
		return -1;
	return send_dsp(comm, dat >> 8);
}

/*	c m d _ d s p
 *
 *	Interrupts the DSP and sends a new command. Procedure:
 *		Send the command
 *		Get a "Ready" response from the DSP
 *		Send the command again
 */
int
cmd_dsp(COMM *comm, int cmd)
{
	send_com(comm, cmd, 0);
	if (recv_dsp(comm, 1000) != 'R')
		return -1;
	send_com(comm, cmd, 0);
	return 0;
}

/*	c m d _ r e s p
 *
 *	Send a command to the DSP and wait "timeout" msec for
 *	a one-byte response.
 */
int
cmd_resp(COMM *comm, int cmd, unsigned timeout)
{
	if ((cmd_dsp(comm, cmd)) == -1)
		return -1;
	return recv_dsp(comm, timeout);
}

/*	c m d _ r e s p 1 6
 *
 *	Send a command to the DSP and wait "timeout" msec for
 *	a two-byte response.
 */
long
cmd_resp16(COMM *comm, int cmd, unsigned timeout)
{
	long l, h;

	if ((l = cmd_resp(comm, cmd, timeout)) == -1)
		return -1;
	if ((h = recv_dsp(comm, 100)) == -1)
		return -1;
	return (h << 8) | l;
}

long
cmd_resp32(COMM *comm, int cmd, unsigned timeout)
{
	long l, h;

	if ((l = cmd_resp16(comm, cmd, timeout)) == -1)
		return -1;
	if ((h = recv_dsp(comm, 100)) == -1)
		return -1;
	l = ((l & 0x7fff) << 16) | h;
	if ((h = recv_dsp(comm, 100)) == -1)
		return -1;
	l = l | (h << 8);
	return l;
}

long
cmd_resp64(COMM *comm, int cmd, unsigned timeout, long *sec)
{
	long l, h;

	if ((l = cmd_resp32(comm, cmd, timeout)) == -1)
		return -1;
	if (sec != NULL)
		*sec = l;
	if ((l = recv_dsp(comm, 100)) == -1)
		return -1;
	if ((h = recv_dsp(comm, 100)) == -1)
		return -1;
	l = (h << 8) | l;
	if ((h = recv_dsp(comm, 100)) == -1)
		return -1;
	l = ((l & 0x7fff) << 16) | h;
	if ((h = recv_dsp(comm, 100)) == -1)
		return -1;
	l = l | (h << 8);
	return l;
}
