from myhdl import *
from config import *
from math import ceil, log

def counter(reset, clk, counterSig, externalSig, start, done, mode, interval,
			gate, sigSelect, count):
	"""
	Inputs:
		reset
		clk
		counterSig		Signal to be counted (external signal or clock)
		externalSig		Signal to be measured
		start
		done
		mode			Measurement mode: 0 - frequency, 1 - period
		interval[2:]	Measurement interval: 0, 1, 2.
							Frequency: 0.1, 1.0 or 10.0 seconds
							Period: 1, 10 or 100 cycles
	Outputs:
		gate[2:]		bit 1: counter signal, bit 0: timer (external) signal.
						Positive logic (high enables)
		sigSelect		1 - clock sig input to counter; 0 - external sig input to counter
		count[32:]		Measurement results

	mode=0: The frequency of the external signal is measured:
			The external signal is switched to the counterSig input pin
			The measurement interval is determined by the system clock.

	mode=1:	The period of the external signal is measured:
			The clock signal is switched to the counterSig input pin.
			The external signal is switched to the timerSig input pin.
			The period of the external signal is determined by counting the number of
			clock cycles in the measurement interval.
			The measurement interval is determined by counting the cycles
			of the external signal. The external signal frequency must be
			less than the clock frequency.
	"""

	State = enum('STATE1', 'STATE2', 'STATE3', 'STATE4',
				 'STATE5', 'STATE5A', 'STATE6', 'STATE7', 'STATE8')
	state = Signal(State.STATE1)
	freqIntervals = tuple([int(MASTER_CLOCK_FREQUENCY/10), int(MASTER_CLOCK_FREQUENCY),
							int(MASTER_CLOCK_FREQUENCY*10)])
	periodIntervals = tuple([1, 10, 100])
	ixf = Signal(intbv(val=0, min=0, max=MASTER_CLOCK_FREQUENCY*10))
	ixp = Signal(intbv(val=0, min=0, max=100))
	exSig, exSig1, exSigState, counterReset = [Signal(bool(0)) for k in range(4)]

	@always(clk.negedge)
	def exSigLogic():
		exSig1.next = externalSig
		exSig.next = exSig1

	@always(reset.negedge, clk.negedge)
	def clockLogic():
		# Count clock or external signal ticks to establish the measurement interval
		if reset == ACTIVE_LOW:
			done.next = INACTIVE_HIGH
			gate.next = GATE_CLOSED
			counterReset.next = INACTIVE_HIGH
			state.next = State.STATE1
		else:
			if state == State.STATE1:
				if start == ACTIVE_LOW:
					if mode == FREQUENCY_MODE:
						sigSelect.next = 0		# route external signal to counter
						ixf.next = freqIntervals[interval]
					else:
						sigSelect.next = 1		# route 10- MHz clock to counter
						ixp.next = periodIntervals[interval]
					done.next = INACTIVE_HIGH
					counterReset.next = ACTIVE_LOW
					state.next = State.STATE2

			elif state == State.STATE2:
				counterReset.next = INACTIVE_HIGH
				state.next = State.STATE3

			elif state == State.STATE3:
				if mode == FREQUENCY_MODE:
					gate.next[0] = GATE_OPEN	# start counting external signal.
					state.next = State.STATE4
				else:
					gate.next[1] = GATE_OPEN	# enable external signal to timer (this loop)
					state.next = State.STATE5

			elif state == State.STATE4:
				# frequency measurement loop
				if ixf <= 1:
					gate.next = GATE_CLOSED
					done.next = ACTIVE_LOW
					state.next = State.STATE1
				else:
					ixf.next = ixf - 1

			elif state == State.STATE5:
				# States 5 - 6 ensure that the period timing is determined by full cycles
				# of the external signal (i.e., does not start on a partial cycle)
				exSigState.next = 0
				if exSig == 0:
					state.next = State.STATE5A

			elif state == State.STATE5A:
				if exSig == 1:
					# posedge on exSig detected
					state.next = State.STATE6

			elif state == State.STATE6:
				if exSig == 0:
					# negedge on exSig detected
					gate.next[0] = GATE_OPEN	# start counting clock ticks
					state.next = State.STATE7

			elif state == State.STATE7:
				# period measurement loop
				if exSigState == 0:
					if exSig == 0:
						if ixp == 0:
							gate.next = GATE_CLOSED		# close both gates
							done.next = ACTIVE_LOW
							state.next = State.STATE1
						else:
							# Count negative transitions of the external signal.
							ixp.next = ixp - 1
							exSigState.next = 1
				else:
					if exSig == 1:
						exSigState.next = 0
			else:
				state.next = State.STATE1

	@always(counterReset.negedge, counterSig.negedge)
	def counterLogic():
		if counterReset == ACTIVE_LOW:
			# Synchronization problems with reset and count (due to different clock domains) do not arise in this case
			# because counterSig is not active when the reset occurs or when count is accessed by the clk domain.
			count.next = 0
		else:
			count.next = count + 1

	return instances()

def test1():
	reset, clk, counterSig, externalSig, start, done, mode = [Signal(bool(0)) for k in range(7)]
	sigSelect = Signal(bool(0))
	interval, gate = [Signal(intbv(0)[2:]) for k in range(2)]
	count = Signal(intbv(0)[COUNT_WIDTH:])
	toVerilog(counter, reset, clk, counterSig, externalSig, start, done, mode, interval, gate, sigSelect, count)

if __name__ == "__main__":
	test1()

